/*
 * Copyright (C) GridGain Systems. All Rights Reserved.
 * _________        _____ __________________        _____
 * __  ____/___________(_)______  /__  ____/______ ____(_)_______
 * _  / __  __  ___/__  / _  __  / _  / __  _  __ `/__  / __  __ \
 * / /_/ /  _  /    _  /  / /_/ /  / /_/ /  / /_/ / _  /  _  / / /
 * \____/   /_/     /_/   \_,__/   \____/   \__,_/  /_/   /_/ /_/
 *
 */

package org.gridgain.examples.snapshots;

import java.util.Collections;
import org.apache.ignite.Ignite;
import org.apache.ignite.IgniteCache;
import org.apache.ignite.IgniteDataStreamer;
import org.apache.ignite.Ignition;
import org.gridgain.grid.GridGain;
import org.gridgain.grid.persistentstore.SnapshotFuture;

/**
 * This example demonstrates usage of cluster-wide snapshots in GridGain.
 * <p>
 * To execute this example you should start an instance of {@code SnapshotsExampleNodeStartup}
 * class which will start up a GridGain remote server node with a proper configuration.
 * <p>
 * A directory for snapshot will be created in the {@code IGNITE_HOME/work/snapshot} folder.
 * <p>
 * Note, to clean previously created snapshot folders, you should delete
 * the {@code IGNITE_HOME/work/db} folder.
 */
public class SnapshotsExample {
    /** */
    private static final int COUNT = 500;

    /**
     * @param args Program arguments, ignored.
     * @throws Exception If failed.
     */
    public static void main(String[] args) throws Exception {
        Ignition.setClientMode(true);

        try (Ignite ig = Ignition.start("config/snapshots-example.xml")) {
            // Activate the cluster. Required to do if the persistent store is enabled because you might need
            // to wait while all the nodes, that store a subset of data on disk, join the cluster.
            ig.active(true);

            GridGain gg = ig.plugin(GridGain.PLUGIN_NAME);

            System.out.println("Populating the cache...");

            try (IgniteDataStreamer<Integer, Organization> streamer = ig.dataStreamer("Org")) {
                streamer.allowOverwrite(true);

                for (int i = 0; i < COUNT; i++) {
                    streamer.addData(i, new Organization(i, "Organization-" + i));

                    if (i > 0 && i % 1_000 == 0)
                        System.out.println("Done: " + i);
                }
            }

            System.out.println("Starting snapshot for organization cache...");

            String msg = "[src=example, user=" + System.getProperty("user.name") + "]";

            SnapshotFuture snapshotFuture = gg.snapshot().createFullSnapshot(
                Collections.singleton("Org"), msg);

            snapshotFuture.get();

            System.out.println("Snapshot has been saved.");

            System.out.println("Overwriting the cache...");

            try (IgniteDataStreamer<Integer, Organization> streamer = ig.dataStreamer("Org")) {
                streamer.allowOverwrite(true);

                for (int i = 0; i < COUNT; i++) {
                    streamer.addData(i, new Organization(i, "Modified-Organization-" + i));

                    if (i > 0 && i % 1_000 == 0)
                        System.out.println("Done: " + i);
                }
            }

            SnapshotFuture incrementalSnapshotFuture = gg.snapshot()
                .createSnapshot(Collections.singleton("Org"), msg);

            incrementalSnapshotFuture.get();

            System.out.println("Restoring from full snapshot...");

            gg.snapshot().restoreSnapshot(snapshotFuture.snapshotOperation().snapshotId(),
                Collections.singleton("Org"), msg).get();

            System.out.println("Validating full snapshot...");

            IgniteCache<Integer, Organization> cache = ig.cache("Org");

            for (int i = 0; i < COUNT; i++) {
                Organization organization = cache.get(i);

                if (organization == null) {
                    System.out.println("Failed to restore organization " + i + " from snapshot.");

                    return;
                }

                if (organization.getId() != i || !("Organization-" + i).equals(organization.getName())) {
                    System.out.println("Organization " + i + " has invalid value: " + organization);

                    return;
                }
            }

            System.out.println("Full snapshot has been successfully restored.");

            System.out.println("Restoring from incremental snapshot...");

            gg.snapshot().restoreSnapshot(incrementalSnapshotFuture.snapshotOperation().snapshotId(),
                Collections.singleton("Org"), msg).get();

            System.out.println("Validating incremental snapshot...");

            cache = ig.cache("Org");

            for (int i = 0; i < COUNT; i++) {
                Organization organization = cache.get(i);

                if (organization == null) {
                    System.out.println("Failed to restore organization " + i + " from snapshot.");

                    return;
                }

                if (organization.getId() != i || !("Modified-Organization-" + i).equals(organization.getName())) {
                    System.out.println("Organization " + i + " has invalid value: " + organization);

                    return;
                }
            }

            System.out.println("Incremental snapshot has been successfully restored.");
        }
    }
}
