/*
 *  Copyright (C) GridGain Systems. All Rights Reserved.
 *  _________        _____ __________________        _____
 *  __  ____/___________(_)______  /__  ____/______ ____(_)_______
 *  _  / __  __  ___/__  / _  __  / _  / __  _  __ `/__  / __  __ \
 *  / /_/ /  _  /    _  /  / /_/ /  / /_/ /  / /_/ / _  /  _  / / /
 *  \____/   /_/     /_/   \_,__/   \____/   \__,_/  /_/   /_/ /_/
 */

package org.gridgain.examples.dr;

import org.apache.ignite.Ignite;
import org.apache.ignite.IgniteCache;
import org.apache.ignite.Ignition;
import org.apache.ignite.events.EventType;
import org.gridgain.examples.dr.misc.ChronologicalConflictResolver;
import org.gridgain.examples.dr.misc.Dr1NodeStartup;
import org.gridgain.examples.dr.misc.Dr2NodeStartup;
import org.gridgain.examples.dr.misc.DrUpdatesListener;
import org.gridgain.examples.dr.misc.Value;

/**
 * The example demonstrates active-active replication between two separate data centers.
 * Each data center updates cache's content in parallel and replicates changes to each other.
 * <p>
 * To execute the example you need to start {@link Dr2HubStartup} that will launch a sender/receiver HUB node belonging
 * to the remote (second) data center.
 * Once it's started the sender HUB from the first data center, started using this {@code DrExample} executable, will
 * be able to connect to the remote receiver hub from the second data center and the replicas will go from both
 * directions.
 * Until both data centers get connected you will see a harmless error message in the log containing the following
 * '[DrSenderRemoteDataCenterNode] Could not connect to replica...'
 * <p>
 * To trigger that replicas are received successfully each node from every data center registers
 * {@link DrUpdatesListener} printing out only those updates to the logs that are received from the remote side.
 * <p>
 * Both data centers update the same set of keys and {@link ChronologicalConflictResolver} is used to determine
 * which entry version should be stored in the cache of a particular data center - local or the one received from the
 * remote DR.
 * <p>
 * Finally, you may want to start additional nodes that belong to the first data center using {@link Dr1NodeStartup}
 * and to the second data center using {@link Dr2NodeStartup} to see that they also receive replicas.
 */
public class DrExample {
    /** Cache name */
    static final String CACHE_NAME = "crossDrCache";

    /**
     * @param args Program arguments, ignored.
     * @throws Exception If failed.
     */
    public static void main(String[] args) throws Exception {
        // Staring a node from first data center that is both receiver and sender HUB.
        try (Ignite dr1Hub = Ignition.start("config/dr/dr1-sender-receiver-hub.xml")) {
            // Listening for cache update printing out those that come from the remote data center prefixed 'dr2_'.
            dr1Hub.events().localListen(new DrUpdatesListener("dr2"), EventType.EVT_CACHE_OBJECT_PUT);
            
            IgniteCache<Integer, Value> cache = dr1Hub.cache(CACHE_NAME);

            long valGen = 0;

            /*
             * Until sender HUB connects to the remote data center you'll see harmless error message in the log
             * '[DrSenderRemoteDataCenterNode] Could not connect to replica...'
             */
            while (true) {
                for (int i = 0; i < 500; i++) {
                    /*
                     * Overriding a value in the cache setting the prefix of local data center.
                     * The remote data center will trigger this using {@link DrUpdatesListener}.
                     *
                     * When a value that has the same key is updated by both data centers, then
                     * {@link ChronologicalConflictResolver} is used to decide which value to keep in the cache of each
                     * data center.
                     */
                    cache.put(i, new Value("dr1_" + valGen++));

                    Thread.sleep(1500);
                }
            }
        }
    }
}
