#!/usr/bin/env bash

#
# Copyright (C) GridGain Systems. All Rights Reserved.
#  _________        _____ __________________        _____
#  __  ____/___________(_)______  /__  ____/______ ____(_)_______
#  _  / __  __  ___/__  / _  __  / _  / __  _  __ `/__  / __  __ \
#  / /_/ /  _  /    _  /  / /_/ /  / /_/ /  / /_/ / _  /  _  / / /
#  \____/   /_/     /_/   \_,__/   \____/   \__,_/  /_/   /_/ /_/
#

#
# GridGain Control Center command line loader.
#

#
# The function exports WORK_FOLDER variable with path to GridGain Control Center installation folder.
#
setJarFile() {
    WORK_FOLDER="$(cd "$(dirname "$0")"; "pwd")";

    JAR_FILE=${WORK_FOLDER}/libs/gridgain-cloud-connector-2025.4-7.jar

    #
    # Check WORK_FOLDER is valid.
    #
    if [ ! -f "${JAR_FILE}" ]; then
        echo $0", ERROR:"
        echo "Connector installation folder is incorrect."

        exit 1
    fi
}

setJarFile

javaVersion() {
    version=$("$1" -version 2>&1 | awk -F '"' '/version/ {print $2}')
}

# Extract only major version of java to `version` variable.
javaMajorVersion() {
    javaVersion "$1"
    version="${version%%.*}"

    if [ $version -eq 1 ]; then
        # Version seems starts from 1, we need second number.
        javaVersion "$1"
        backIFS=$IFS

        IFS=. ver=(${version##*-})
        version=${ver[1]}

        IFS=$backIFS
    fi
}

#
# Discovers path to Java executable and checks it's version.
# The function exports JAVA variable with path to Java executable.
#
checkJava() {
    # Check JAVA_HOME.
    if [ "${JAVA_HOME:-}" = "" ]; then
        JAVA=`type -p java`
        RETCODE=$?

        if [ $RETCODE -ne 0 ]; then
            echo $0", ERROR:"
            echo "JAVA_HOME environment variable is not found."
            echo "Please point JAVA_HOME variable to location of JDK 17 or later."

            exit 1
        fi

        JAVA_HOME=
    else
        JAVA=${JAVA_HOME}/bin/java
    fi

    #
    # Check JDK.
    #
    javaMajorVersion "$JAVA"

    if [ $version -lt 17 ]; then
        echo "$0, ERROR:"
        echo "The $version version of JAVA installed in JAVA_HOME=$JAVA_HOME is incompatible."
        echo "Please point JAVA_HOME variable to installation of JDK 17 or later."
        exit 1
    fi
}

#
# JVM options. See http://java.sun.com/javase/technologies/hotspot/vmoptions.jsp for more details.
#
# ADD YOUR/CHANGE ADDITIONAL OPTIONS HERE
#
if [ -z "$JVM_OPTS" ] ; then
    JVM_OPTS="-Xms1g -Xmx2g -server -XX:MaxMetaspaceSize=256m -XX:MaxDirectMemorySize=1g"
fi

#
# Uncomment the following GC settings if you see spikes in your throughput due to Garbage Collection.
#
#JVM_OPTS="$JVM_OPTS -XX:+UseG1GC"

#
# Uncomment if you get StackOverflowError.
# On 64 bit systems this value can be larger, e.g. -Xss16m
#
# JVM_OPTS="${JVM_OPTS} -Xss4m"

#
# Uncomment to set preference for IPv4 stack.
#
# JVM_OPTS="${JVM_OPTS} -Djava.net.preferIPv4Stack=true"

#
# Function that adds specified value as JVM property if it wasn't already set
#
addJvmProperty() {
    local prop="${1}"
    local value="${2}"

    case "${JVM_OPTS:-}" in
        *-D${prop}=*)
            ;;
        *)
            JVM_OPTS="${JVM_OPTS:-} -D${prop}=${value}"
            ;;
    esac
}

#
# Set 'file.encoding' to UTF-8 default if not specified otherwise
#
addJvmProperty "file.encoding" "UTF-8"

#
# Discover path to Java executable and check it's version.
#
checkJava

JVM_OPTS="\
   --add-opens=java.base/jdk.internal.access=ALL-UNNAMED \
   --add-opens=java.base/jdk.internal.misc=ALL-UNNAMED \
   --add-opens=java.base/sun.nio.ch=ALL-UNNAMED \
   --add-opens=java.management/com.sun.jmx.mbeanserver=ALL-UNNAMED \
   --add-opens=jdk.internal.jvmstat/sun.jvmstat.monitor=ALL-UNNAMED \
   --add-opens=java.base/sun.reflect.generics.reflectiveObjects=ALL-UNNAMED \
   --add-opens=jdk.management/com.sun.management.internal=ALL-UNNAMED \
   --add-opens=java.base/java.io=ALL-UNNAMED \
   --add-opens=java.base/java.net=ALL-UNNAMED \
   --add-opens=java.base/java.nio=ALL-UNNAMED \
   --add-opens=java.base/java.security.cert=ALL-UNNAMED \
   --add-opens=java.base/java.util=ALL-UNNAMED \
   --add-opens=java.base/java.util.concurrent=ALL-UNNAMED \
   --add-opens=java.base/java.util.concurrent.locks=ALL-UNNAMED \
   --add-opens=java.base/java.util.concurrent.atomic=ALL-UNNAMED \
   --add-opens=java.base/java.lang=ALL-UNNAMED \
   --add-opens=java.base/java.lang.invoke=ALL-UNNAMED \
   --add-opens=java.base/java.math=ALL-UNNAMED \
   --add-opens=java.base/java.time=ALL-UNNAMED \
   --add-opens=java.base/sun.security.ssl=ALL-UNNAMED \
   --add-opens=java.base/sun.security.x509=ALL-UNNAMED \
   --add-opens=java.sql/java.sql=ALL-UNNAMED \
  ${JVM_OPTS}"

"$JAVA" -Dlogging.config="${WORK_FOLDER}/log4j2.xml" ${JVM_OPTS} -jar ${JAR_FILE}
